import { navLinks } from "../constants";
import { Link } from "react-router-dom";
import { useSelector } from "react-redux";
import { HiOutlineMenuAlt3 } from "react-icons/hi";
import { IoClose } from "react-icons/io5";
import { FiUser } from "react-icons/fi";
import { useState, useEffect } from "react";
import { Drawer } from "antd";

function Header() {
  const { currentUser } = useSelector((state) => state.user);
  const [nav, setNav] = useState(false);
  const [scrolled, setScrolled] = useState(false);

  // Handle scroll effect
  useEffect(() => {
    const handleScroll = () => {
      setScrolled(window.scrollY > 50);
    };
    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);

  return (
    <header
      className={`fixed top-0 left-0 right-0 z-50 transition-all duration-300 ${
        scrolled
          ? "bg-white/90 backdrop-blur-lg shadow-soft py-3"
          : "bg-transparent py-5"
      }`}
    >
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex justify-between items-center">
          {/* Logo */}
          <Link to="/" className="flex items-center gap-2">
            <div className="w-10 h-10 bg-gradient-to-br from-primary-500 to-primary-600 rounded-xl flex items-center justify-center shadow-lg">
              <span className="text-white font-bold text-xl">R</span>
            </div>
            <span className="text-xl font-bold text-secondary-900 hidden sm:block">
              Rent<span className="text-primary-500">Ride</span>
            </span>
          </Link>

          {/* Desktop Navigation */}
          <nav className="hidden lg:flex items-center gap-8">
            {navLinks.map((navlink, index) => (
              <Link
                key={index}
                to={navlink.path}
                className="nav-link text-secondary-700 font-medium hover:text-primary-600 transition-colors"
              >
                {navlink.title}
              </Link>
            ))}
          </nav>

          {/* Auth Buttons */}
          <div className="hidden lg:flex items-center gap-3">
            {currentUser && !currentUser.isAdmin && !currentUser.isVendor ? (
              <Link to="/profile" className="flex items-center gap-2">
                {currentUser.profilePicture ? (
                  <img
                    src={currentUser.profilePicture}
                    alt="Profile"
                    referrerPolicy="no-referrer"
                    className="w-10 h-10 rounded-full object-cover ring-2 ring-primary-500 ring-offset-2"
                  />
                ) : (
                  <div className="w-10 h-10 rounded-full bg-primary-100 flex items-center justify-center">
                    <FiUser className="text-primary-600 text-xl" />
                  </div>
                )}
                <span className="font-medium text-secondary-700">
                  {currentUser.username}
                </span>
              </Link>
            ) : (
              <>
                <Link to="/signIn">
                  <button className="px-5 py-2.5 text-secondary-700 font-medium hover:text-primary-600 transition-colors">
                    Sign In
                  </button>
                </Link>
                <Link to="/signup">
                  <button className="px-5 py-2.5 bg-gradient-to-r from-primary-500 to-primary-600 text-white font-semibold rounded-xl shadow-lg hover:shadow-glow transition-all duration-300 hover:-translate-y-0.5">
                    Get Started
                  </button>
                </Link>
              </>
            )}
          </div>

          {/* Mobile Menu Button */}
          <button
            onClick={() => setNav(!nav)}
            className="lg:hidden w-10 h-10 flex items-center justify-center rounded-xl bg-gray-100 hover:bg-gray-200 transition-colors"
          >
            {nav ? (
              <IoClose className="text-2xl text-secondary-700" />
            ) : (
              <HiOutlineMenuAlt3 className="text-2xl text-secondary-700" />
            )}
          </button>
        </div>
      </div>

      {/* Mobile Drawer */}
      <Drawer
        placement="right"
        onClose={() => setNav(false)}
        open={nav}
        width={300}
        className="lg:hidden"
        styles={{
          body: { padding: 0 },
        }}
      >
        <div className="flex flex-col h-full bg-white">
          {/* Drawer Header */}
          <div className="p-6 border-b border-gray-100">
            <Link to="/" className="flex items-center gap-2" onClick={() => setNav(false)}>
              <div className="w-10 h-10 bg-gradient-to-br from-primary-500 to-primary-600 rounded-xl flex items-center justify-center">
                <span className="text-white font-bold text-xl">R</span>
              </div>
              <span className="text-xl font-bold text-secondary-900">
                Rent<span className="text-primary-500">Ride</span>
              </span>
            </Link>
          </div>

          {/* Nav Links */}
          <div className="flex-1 py-6">
            {navLinks.map((navlink, index) => (
              <Link
                key={index}
                to={navlink.path}
                className="flex items-center px-6 py-4 text-lg font-medium text-secondary-700 hover:bg-primary-50 hover:text-primary-600 transition-colors"
                onClick={() => setNav(false)}
              >
                {navlink.title}
              </Link>
            ))}
          </div>

          {/* Auth Section */}
          <div className="p-6 border-t border-gray-100 space-y-3">
            {currentUser && !currentUser.isAdmin && !currentUser.isVendor ? (
              <Link
                to="/profile"
                className="flex items-center gap-3 p-3 rounded-xl bg-gray-50"
                onClick={() => setNav(false)}
              >
                {currentUser.profilePicture ? (
                  <img
                    src={currentUser.profilePicture}
                    alt="Profile"
                    className="w-12 h-12 rounded-full object-cover"
                  />
                ) : (
                  <div className="w-12 h-12 rounded-full bg-primary-100 flex items-center justify-center">
                    <FiUser className="text-primary-600 text-xl" />
                  </div>
                )}
                <div>
                  <p className="font-semibold text-secondary-900">{currentUser.username}</p>
                  <p className="text-sm text-gray-500">View Profile</p>
                </div>
              </Link>
            ) : (
              <>
                <Link to="/signIn" onClick={() => setNav(false)}>
                  <button className="w-full py-3 text-secondary-700 font-semibold border-2 border-gray-200 rounded-xl hover:border-primary-500 transition-colors">
                    Sign In
                  </button>
                </Link>
                <Link to="/signup" onClick={() => setNav(false)}>
                  <button className="w-full py-3 bg-gradient-to-r from-primary-500 to-primary-600 text-white font-semibold rounded-xl shadow-lg">
                    Get Started
                  </button>
                </Link>
              </>
            )}
          </div>
        </div>
      </Drawer>
    </header>
  );
}

export default Header;
